--Kishore Padmaraju
--April 9th, 2013
-- FSM for Mouse

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.std_logic_unsigned.all;

entity Mouse_FSM is
	port(
		CLOCK		: in std_logic; 						-- 50 MHz clock 
		FSM_en		: in std_logic; 						-- FSM_en is to enable or halt the FSM		
		GPIO		: inout std_logic_vector(35 downto 0); 	-- GPIO pin connections
		MEM_wr		: out std_logic_vector(4 downto 0);		-- indicates whether data is being written to RAM, dx, dy, lc, or rc registers
		data_out	: out std_logic_vector(7 downto 0);	-- data written out to memory blocks
		wr_addr		: out std_logic_vector(7 downto 0)
	);
end Mouse_FSM;

architecture FSM of Mouse_FSM is
	type STATE_TYPE is (IDLE, W10, W11, 
	CF0, CF1, CF2, CF3, CF4, CF5, CF6, CF7, CF8, CF9, CF10, 
	M0, M1, M2, M3, M4, M5, M6, M7, M8, M9, 
	DX0, DX1, DX2, DX3, DX4, DX5, DX6, DX7, DX8, DX9, 
	DY0, DY1, DY2, DY3, DY4, DY5, DY6, DY7, DY8, DY9, 
	PxD0, PxD1, PxD2, PxD3, PxD4, PxD5, PxD6, PxD7, PxD8, PxD9, PxD10, PxD11, 
	W0, W1, W2, W3, W4, W5, W6, W7, W8, W9, LC, RC, N1);
	signal Y: STATE_TYPE := IDLE;
	
	signal SDIO_en: std_logic := '1'; -- sets GPIO pin to read/write (set 0/1) for SDIO
	signal SCLK_en: std_logic := '1'; -- sets GPIO pin to read/write (set 0/1) for SCLK
	signal PD_en: 	std_logic := '1'; -- sets GPIO pin to read/write (set 0/1) for PD
	
	signal SDIO: 	std_logic := '1';        
	signal SCLK: 	std_logic := '1'; -- SCLK is clock driving communication with Optical processor
	signal PD: 		std_logic := '0'; -- PD is power-down pin, set to 0 to keep Optical processor constantly on
	
	constant counter_size_CLK: 		integer := 7; -- 8 bits, counter will stop at "10000000", yielding a clock rate of 97.65 kHz
	constant counter_size_100us: 	integer := 5;  -- 5 bits, counter will stop at "10000", yielding a wait of 163.8 us (> 100 us)
	signal CLK: 					std_logic := '1'; -- Slow clock driving state transitions (has to be < 4 MHz to work with optical processor chip)
	signal counter_CLK: 			unsigned(counter_size_CLK downto 0) := (others => '0'); -- counter to generate slow clock from 50 MHz clock
	signal counter_100us: 			unsigned(counter_size_100us downto 0) := (others => '0'); --  counter to wait at least 100us
	signal counter_mem:				unsigned(counter_size_CLK downto 0) := (others => '0'); -- counter for waiting between memory writes
	
	constant CF_reg_addr: 	std_logic_vector(7 downto 0) := X"0A"; -- address of configuration register 
	constant MOT_reg_addr: 	std_logic_vector(7 downto 0) := X"02"; -- address of motion register 
	constant DX_reg_addr: 	std_logic_vector(7 downto 0) := X"03"; -- address of dx coordinate register 
	constant DY_reg_addr: 	std_logic_vector(7 downto 0) := X"04"; -- address of dy coordinate register
	constant DATA_reg_addr: std_logic_vector(7 downto 0) := X"0C"; -- address of data_lower register (register that holds the pixel value)	
	constant CF_reg_dft:	std_logic_vector(7 downto 0) := X"01"; -- configuration register default setting, sleep mode turned off
	constant CF_reg_pxd: 	std_logic_vector(7 downto 0) := X"09"; -- configuration register pixel-dump setting
	
	signal pxd_en: 		std_logic 					 := '0';			 -- flag indicating pixel dump
	signal px_val:		std_logic_vector(7 downto 0) := (others => '0');
	signal px_addr: 	unsigned(7 downto 0) 		 := (others => '0');
	signal MOT_reg_val: std_logic_vector(7 downto 0) := (others => '0'); -- holds the value of the read motion register, the MSB indicates motion occured
	signal DX_reg_val: 	std_logic_vector(7 downto 0) := (others => '0'); -- holds the value of the read dx coordinate register
	signal DY_reg_val: 	std_logic_vector(7 downto 0) := (others => '0'); -- holds the value of the read dy coordinate register
	signal bit_num:		unsigned(3 downto 0);							 -- keeps track of bit numbers when reading or writing registers
	signal LC_val:	 	std_logic					 := '0';			 -- high-value indicates left-click during image sample
	signal RC_val:		std_logic					 := '0'; 			 -- high-value indicates right-click during image sample	
	
	signal mem_write: 	std_logic_vector(4 downto 0) := "00000"; -- flags indicating memory writes, "00001":dx coordinate, "010":dy coordinate, "100":pixel value
	signal mem_ff1	:	std_logic_vector(4 downto 0) := "00000";  -- flip-flops for detecting changes in memory write conditions
	signal mem_ff2  : 	std_logic_vector(4 downto 0) := "00000";

begin
		
	with SDIO_en select
		GPIO(0) <= 'Z' when '0', SDIO when others;
	with SCLK_en select
		GPIO(1) <= 'Z' when '0', SCLK when others;
	with PD_en select
		GPIO(2) <= 'Z' when '0', PD when others;
		
	GPIO(3) <= 'Z'; -- left-click
	GPIO(4) <= 'Z'; -- right-click
	
	GPIO(5) <= CLK;
	
	Write_to_Memory: process(CLOCK)
	begin
		if(rising_edge(CLOCK)) then

			mem_ff1 <= mem_write;
			mem_ff2 <= mem_ff1; 
			
			MEM_wr <= mem_ff1 and (not mem_ff2); 
			
			case mem_write is
				when "00001" =>
					wr_addr <= x"00";
					-- convert 8-bit 2's complement to 16-bit 2's complement by repeating the MSB
					data_out <= DX_reg_val;
				when "00010" =>
					wr_addr <= x"00";
					-- convert 8-bit 2's complement to 16-bit 2's complement by repeating the MSB
					data_out <= DY_reg_val;
				when "00100" =>
					wr_addr <= std_logic_vector(px_addr); 
					data_out <= px_val;
				when "01000" =>
					wr_addr <= x"00";
					data_out <= (7 downto 1 => '0') & LC_val;
				when "10000" =>
					wr_addr <= x"00";
					data_out <= (7 downto 1 => '0') & RC_val;
				when others =>
					wr_addr <= x"00";
					data_out <= (others => '0');
			end case;
		end if;
	end process Write_to_Memory;

	Slow_Clock: process(CLOCK)
	begin
		if(rising_edge(CLOCK)) then
			if(counter_CLK=x"F9") then
				CLK <= NOT CLK;
				counter_CLK <= (others => '0');
			else
				counter_CLK <= counter_CLK + 1;
			end if;
		end if;
	end process Slow_Clock;
		
	process(CLOCK)
	begin
		if(rising_edge(CLOCK)) then
		case Y is
			when IDLE =>
				if(CLK='0' AND FSM_en='1') then
					Y <= W10;
				end if;
			when W10 =>
				if(CLK='1') then
					Y <= W11;
				end if;
			when W11 =>
				if(CLK ='0') then
					
					if(counter_100us="001010") then
						Y <= CF0;
					else
						Y <= W10;
					end if;
				end if;
			when CF0 =>
				if(CLK='1') then
					Y <= CF1;
				end if;
			when CF1 =>
				if(CLK='0') then
					Y <= CF2;
				end if;
			when CF2 =>
				if(CLK='1') then
					Y <= CF3;
				end if;
			when CF3 =>
				if(CLK='0') then
					if(bit_num=X"0") then
						Y <= CF5;
					else
						Y <= CF4;
					end if;
				end if;
			when CF4 =>
				if(CLK='1') then
					Y <= CF3;
				end if;
			when CF5 =>
				if(CLK='1') then
					if(pxd_en='0') then
						Y <= CF6;
					elsif(pxd_en='1') then
						Y <= CF8;
					end if;
				end if;
			when CF6 =>
				if(CLK='0') then
					if(bit_num=X"0") then
						Y <= CF10;
					else
						Y <= CF7;
					end if;
				end if;
			when CF7 =>
				if(CLK='1') then
					Y <= CF6;
				end if;
			when CF8 =>
				if(CLK='0') then
					if(bit_num=X"0") then
						Y <= CF10;
					else
						Y <= CF9;
					end if;
				end if;
			when CF9 =>
				if(CLK='1') then
					Y <= CF8;
				end if;
			when CF10 =>
				if(CLK='1') then
					Y <= W0;
				end if;
			when W0 =>
				if(CLK='0') then
					Y <= W1;
				end if;
			when W1 =>
				if(CLK='1') then
					
					if(counter_100us="001010") then
						if(pxd_en='0') then
							Y <= M0;
						elsif(pxd_en='1') then
							Y <= PxD0;
						end if;
					else
						Y <= W0;
					end if;
				end if;
			when M0 =>
				if(CLK='0') then
					Y <= M1;
				end if;
			when M1 =>
				if(CLK='1') then
					Y <= M2;
				end if;
			when M2 =>
				if(CLK='0') then
					if(bit_num=X"0") then
						Y <= M4;
					else
						Y <= M3;
					end if;
				end if;
			when M3 =>
				if(CLK='1') then
					Y <= M2;
				end if;
			when M4 =>
				if(CLK='1') then
					Y <= M5;
				end if;
			when M5 =>
				if(CLK='0') then
					Y <= W2;
				end if;
			when W2 =>
				if(CLK='1') then
					Y <= W3;
				end if;
			when W3 =>
				if(CLK='0') then
					
					if(counter_100us="001010") then
						Y <= M6;
					else
						Y <= W2;
					end if;
				end if;
			when M6 =>
				if(CLK='1') then
					Y <= M7;
				end if;
			when M7 =>
				if(CLK='0') then
					if(bit_num=X"0") then
						Y <= M9;
					else
						Y <= M8;
					end if;
				end if;
			when M8 =>
				if(CLK='1') then
					Y <= M7;
				end if;
			when M9 =>
				if(CLK='1') then
					if(MOT_reg_val(7)='0') then
						Y <= IDLE;
					elsif(MOT_reg_val(7)='1') then
						Y <= DX0;
					end if;
				end if;
			when DX0 =>
				if(CLK='0') then
					Y <= DX1;
				end if;
			when DX1 =>
				if(CLK='1') then
					Y <= DX2;
				end if;
			when DX2 =>
				if(CLK='0') then
					if(bit_num=X"0") then
						Y <= DX4;
					else
						Y <= DX3;
					end if;
				end if;
			when DX3 =>
				if(CLK='1') then
					Y <= DX2;
				end if;
			when DX4 =>
				if(CLK='1') then
					Y <= DX5;
				end if;
			when DX5 =>
				if(CLK='0') then
					Y <= W4;
				end if;
			when W4 =>
				if(CLK='1') then
					Y <= W5; 
				end if;
			when W5 =>
				if(CLK='0') then
					
					if(counter_100us="001010") then
						Y <= DX6;
					else
						Y <= W4;
					end if;
				end if;
			when DX6 =>
				if(CLK='1') then
					Y <= DX7; 
				end if;
			when DX7 =>
				if(CLK='0') then
					if(bit_num=X"0") then
						Y <= DX9;
					else
						Y <= DX8;
					end if;
				end if;
			when DX8 =>
				if(CLK='1') then
					Y <= DX7;
				end if;
			when DX9 =>
				if(CLK='1') then
					Y <= DY0;
				end if;
			when DY0 =>
				if(CLK='0') then
					Y <= DY1;
				end if;
			when DY1 =>
				if(CLK='1') then
					Y <= DY2;
				end if;
			when DY2 =>
				if(CLK='0') then
					if(bit_num=X"0") then
						Y <= DY4;
					else
						Y <= DY3;
					end if;
				end if;
			when DY3 =>
				if(CLK='1') then
					Y <= DY2;
				end if;
			when DY4 =>
				if(CLK='1') then
					Y <= DY5;
				end if;
			when DY5 =>
				if(CLK='0') then
					Y <= W6;
				end if;
			when W6 =>
				if(CLK='1') then
					Y <= W7; 
				end if;
			when W7 =>
				if(CLK='0') then
					
					if(counter_100us="001010") then
						Y <= DY6;
					else
						Y <= W6;
					end if;
				end if;
			when DY6 =>
				if(CLK='1') then
					Y <= DY7; 
				end if;
			when DY7 =>
				if(CLK='0') then
					if(bit_num=X"0") then
						Y <= DY9;
					else
						Y <= DY8;
					end if;
				end if;
			when DY8 =>
				if(CLK='1') then
					Y <= DY7;
				end if;
			when DY9 =>
				if(CLK='1') then
					Y <= CF0;
				end if;	
			when PxD0 =>
				if(CLK='0') then
					Y <= PxD1;
				end if;
			when PxD1 =>
				if(CLK='1') then
					Y <= PxD2;
				end if;
			when PxD2 =>
				if(CLK='0') then
					if(bit_num=X"0") then
						Y <= PxD4;
					else
						Y <= PxD3;
					end if;
				end if;
			when PxD3 =>
				if(CLK='1') then
					Y <= PxD2;
				end if;
			when PxD4 =>
				if(CLK='1') then
					Y <= PxD5;
				end if;
			when PxD5 =>
				if(CLK='0') then
					Y <= W8;
				end if;
			when W8 =>
				if(CLK='1') then
					Y <= W9;
				end if;
			when W9 =>
				if(CLK='0') then
					
					if(counter_100us="001010") then
						Y <= PxD6;
					else
						Y <= W8;
					end if;
				end if;
			when PxD6 =>
				if(CLK='1') then
					Y <= PxD7;
				end if;
			when PxD7 => 
				if(CLK='0') then
					if(bit_num=X"0") then
						Y <= PxD9;
					else
						Y <= PxD8;
					end if;
				end if;
			when PxD8 =>
				if(CLK='1') then
					Y <= PxD7;
				end if;
			when PxD9 =>
				if(CLK='1') then
					if(px_val(7)='1') then
						Y <= PxD1;
					elsif(px_val(7)='0') then
						if(px_addr=X"FF") then
							Y <= PxD11;
						else
							Y <= PxD10;
						end if;
					end if;
				end if;
			when PxD10 =>
				if(CLK='0') then
					Y <= PxD1;
				end if;
			when PxD11 =>
				if(CLK='0') then
					Y <= LC;
				end if;
			when LC =>
				if(CLK='1') then
					Y <= RC;
				end if;
			when RC =>
				if(CLK='0') then
					Y <= N1;
				end if;
			when N1 =>
				if(CLK='1') then
					Y <= IDLE;
				end if;	
		end case;
		end if;
	end process;
	
	-- Handle setting SDIO pin to write or read
	process(Y)
	begin
		case Y is
			when M5|W2|W3|M6|M7|M8|DX5|W4|W5|DX6|DX7|DX8|DY5|W6|W7|DY6|DY7|DY8|PxD5|W8|W9|PxD6|PxD7|PxD8 =>
				SDIO_en <= '0';
			when others =>
				SDIO_en <= '1';
		end case;
	end process;
	
	-- Handle SCLK generation, writing of SDIO pin
	process(Y)
	begin
		case Y is
			when IDLE =>
				SCLK <= '1';
				SDIO <= '0';
			when W10 =>
				SCLK <= '1';
				SDIO <= '0';
			WHEN W11 =>
				SCLK <= '1';
				SDIO <= '0';
			when CF0 =>
				SCLK <= '1';
				SDIO <= '1';
			when CF1 =>
				SCLK <= '0';
				SDIO <= '1';
			when CF2 =>
				SCLK <= '1';
				SDIO <= '1';
			when CF3 =>
				SCLK <= '0';
				SDIO <= CF_reg_addr(to_integer(bit_num));
			when CF4 =>
				SCLK <= '1';
				SDIO <= CF_reg_addr(to_integer(bit_num));
			when CF5 =>
				SCLK <= '1';
				SDIO <= CF_reg_addr(to_integer(bit_num));
			when CF6 =>
				SCLK <= '0';
				SDIO <= CF_reg_dft(to_integer(bit_num));
			when CF7 =>
				SCLK <= '1';
				SDIO <= CF_reg_dft(to_integer(bit_num));
			when CF8 =>
				SCLK <= '0';
				SDIO <= CF_reg_pxd(to_integer(bit_num));
			when CF9 =>
				SCLK <= '1';
				SDIO <= CF_reg_pxd(to_integer(bit_num));
			when CF10 =>
				SCLK <= '1';
				SDIO <= '1'; 	-- SDIO: Don't Care
			when W0 =>
				SCLK <= '1';
				SDIO <= '1';	-- SDIO: Don't Care
			when W1 =>
				SCLK <= '1';
				SDIO <= '1';	-- SDIO: Don't Care
			when M0 =>
				SCLK <= '0';
				SDIO <= '0';
			when M1 =>
				SCLK <= '1';
				SDIO <= '0';
			when M2 =>
				SCLK <= '0';
				SDIO <= MOT_reg_addr(to_integer(bit_num));
			when M3 =>
				SCLK <= '1';
				SDIO <= MOT_reg_addr(to_integer(bit_num));
			when M4 =>
				SCLK <= '1';
				SDIO <= MOT_reg_addr(to_integer(bit_num));
			when M5 =>
				SCLK <= '1';
				SDIO <= '0';	-- SDIO: Don't Care
			when W2 =>
				SCLK <= '1';
				SDIO <= '0';	-- SDIO: Don't Care
			when W3 =>
				SCLK <= '1';
				SDIO <= '0';	-- SDIO: Don't Care
			when M6 =>
				SCLK <= '0';
				SDIO <= '0';	-- SDIO: Don't Care
			when M7 =>
				SCLK <= '1';
				SDIO <= '0';	-- SDIO: Don't Care
			when M8 =>
				SCLK <= '0';
				SDIO <= '0';	-- SDIO: Don't Care
			when M9 =>
				SCLK <= '1';
				SDIO <= '0';	-- SDIO: Don't Care
			when DX0 =>
				SCLK <= '0';
				SDIO <= '0';
			when DX1 =>
				SCLK <= '1';
				SDIO <= '0';
			when DX2 =>
				SCLK <= '0';
				SDIO <= DX_reg_addr(to_integer(bit_num));
			when DX3 =>
				SCLK <= '1';
				SDIO <= DX_reg_addr(to_integer(bit_num));
			when DX4 =>
				SCLK <= '1';
				SDIO <= DX_reg_addr(to_integer(bit_num));
			when DX5 =>
				SCLK <= '1';
				SDIO <= '0';	-- SDIO: Don't Care
			when W4 =>
				SCLK <= '1';
				SDIO <= '0';	-- SDIO: Don't Care
			when W5 =>
				SCLK <= '1';
				SDIO <= '0';	-- SDIO: Don't Care
			when DX6 =>
				SCLK <= '0';
				SDIO <= '0';	-- SDIO: Don't Care
			when DX7 =>
				SCLK <= '1';
				SDIO <= '0';	-- SDIO: Don't Care
			when DX8 =>
				SCLK <= '0';
				SDIO <= '0';	-- SDIO: Don't Care
			when DX9 =>
				SCLK <= '1';
				SDIO <= '0';	-- SDIO: Don't Care
			when DY0 =>
				SCLK <= '0';
				SDIO <= '0';
			when DY1 =>
				SCLK <= '1';
				SDIO <= '0';
			when DY2 =>
				SCLK <= '0';
				SDIO <= DY_reg_addr(to_integer(bit_num));
			when DY3 =>
				SCLK <= '1';
				SDIO <= DY_reg_addr(to_integer(bit_num));
			when DY4 =>
				SCLK <= '1';
				SDIO <= DY_reg_addr(to_integer(bit_num));
			when DY5 =>
				SCLK <= '1';
				SDIO <= '0';	-- SDIO: Don't Care
			when W6 =>
				SCLK <= '1';
				SDIO <= '0';	-- SDIO: Don't Care
			when W7 =>
				SCLK <= '1';
				SDIO <= '0';	-- SDIO: Don't Care
			when DY6 =>
				SCLK <= '0';
				SDIO <= '0';	-- SDIO: Don't Care
			when DY7 =>
				SCLK <= '1';
				SDIO <= '0';	-- SDIO: Don't Care
			when DY8 =>
				SCLK <= '0';
				SDIO <= '0';	-- SDIO: Don't Care
			when DY9 =>
				SCLK <= '1';
				SDIO <= '0';	-- SDIO: Don't Care
			when PxD0 =>
				SCLK <= '0';
				SDIO <= '0';
			when PxD1 =>
				SCLK <= '1';
				SDIO <= '0';
			when PxD2 =>
				SCLK <= '0';
				SDIO <= DATA_reg_addr(to_integer(bit_num));
			when PxD3 =>
				SCLK <= '1';
				SDIO <= DATA_reg_addr(to_integer(bit_num));
			when PxD4 =>
				SCLK <= '1';
				SDIO <= DATA_reg_addr(to_integer(bit_num));
			when PxD5 =>
				SCLK <= '1';
				SDIO <= '0';	-- SDIO: Don't Care
			when W8 =>
				SCLK <= '1';
				SDIO <= '0';	-- SDIO: Don't Care
			when W9 =>
				SCLK <= '1';
				SDIO <= '0';	-- SDIO: Don't Care
			when PxD6 =>
				SCLK <= '0';
				SDIO <= '0';	-- SDIO: Don't Care
			when PxD7 => 
				SCLK <= '1';
				SDIO <= '0';	-- SDIO: Don't Care
			when PxD8 =>
				SCLK <= '0';
				SDIO <= '0';	-- SDIO: Don't Care
			when PxD9 =>
				SCLK <= '1';
				SDIO <= '0';	-- SDIO: Don't Care
			when PxD10 =>
				SCLK <= '1';
				SDIO <= '0';	-- SDIO: Don't Care
			when PxD11 =>
				SCLK <= '1';
				SDIO <= '0';	-- SDIO: Don't Care
			when LC =>
				SCLK <= '1';
				SDIO <= '0';	-- SDIO: Don't Care
			when RC =>
				SCLK <= '1';
				SDIO <= '0';	-- SDIO: Don't Care
			when N1 =>
				SCLK <= '1';
				SDIO <= '0';	-- SDIO: Don't Care
		end case;
	end process;
	
	-- Handle readingo of GPIO(2) [SDIO} pin
	process(Y)
	begin
		case Y is
			when M7 =>
				MOT_reg_val(to_integer(bit_num)) <= GPIO(0);
			when DX7 =>
				DX_reg_val(to_integer(bit_num)) <= GPIO(0);
			when DY7 =>
				DY_reg_val(to_integer(bit_num)) <= GPIO(0);
			when PxD7 => 
				px_val(to_integer(bit_num)) <= GPIO(0);
			when others =>
				null;
		end case;
	end process;
				
	-- Handle incrementing and resetting of >100 us counter
	process(Y)
	begin
		case Y is
			when W10|W0|W2|W4|W6|W8 =>
				counter_100us <= counter_100us + 1;
			when IDLE|CF10|M5|DX5|DY5|PxD5 =>
				counter_100us <= (others => '0');
			when others =>
				counter_100us <= counter_100us;
		end case;
	end process;

	-- Handle decrementing and resetting of bit_num
	process(Y)
	begin
		case Y is
			when CF4|CF7|CF9|M3|M8|DX3|DX8|DY3|DY8|PxD3|PxD8 =>
				bit_num <= bit_num - 1; 
			when CF2|M1|DX1|DY1|PxD1 =>
				bit_num <= x"6"; -- start at 7th bit-position 
			when CF5|M6|DX6|DY6|PxD6 =>
				bit_num <= x"7"; -- start at 8th bit-position
			when others =>
				null;
		end case;
	end process;

	-- Handle flags for indicating memory write
	process(Y)
	begin
		case Y is
			when DX9 =>
				mem_write <= "00001";
			when DY9 =>
				mem_write <= "00010";
			when PxD9 =>
				mem_write <= "00100";
			when LC =>
				mem_write <= "01000";
			when RC =>
				mem_write <= "10000";
			when others =>
				mem_write <= "00000";
		end case;
	end process;
	
	-- Handle flags for indicated left and right clicks
	process(Y)
	begin
		case Y is
			when LC =>
				lc_val <= GPIO(3);
			when RC =>
				rc_val <= GPIO(4);
			when others =>
				null;
		end case;
	end process;
	
	-- Handle pixel dump flag
	process(Y)
	begin
		case Y is
			when PxD11 =>
				pxd_en <= '0';
			when Dx0 =>
				pxd_en <= '1';
			when others =>
				null;
		end case;
	end process;
	
	-- Handle flags for px_addr incrementing
	process(Y)
	begin
		case Y is
			when PxD0 =>
				px_addr <= X"00";
			when PxD10 =>
				px_addr <= px_addr + 1;
			when others =>
				px_addr <= px_addr;
		end case;
	end process;
	
	--Change state of PD output
	process(Y)
	begin
		case Y is
			when W10|W11 =>
				PD <= '1';
			when others =>
				PD <= '0';
		end case;
		
	end process;
	
end FSM;
